import numpy as np
from functions import get_position_ctr

def alpha_VCG_Mechanism(value_ads, ctr_ads, ctr_org, cvr_ads, cvr_org, alpha, beta):
    """
    Implements the alpha-VCG mechanism for ad auctions.
    
    Parameters:
    - value_ads: The value of ads.
    - ctr_ads: Click-through rate of ads.
    - ctr_org: Click-through rate of organic results.
    - cvr_ads: Conversion rate of ads.
    - cvr_org: Conversion rate of organic results.
    - alpha: Weight for CTR.
    - beta: Weight for CVR.
    
    Returns:
    - click: Average number of clicks.
    - cost: Average cost.
    - cvr: Average conversion rate.
    """
    # Calculate the value for ads and organic results
    value_ads_adjusted = value_ads * ctr_ads + alpha * ctr_ads + beta * cvr_ads
    value_org_adjusted = alpha * ctr_org + beta * cvr_org
    combined_value = np.hstack((value_ads_adjusted, value_org_adjusted))
    
    # Sort the combined values
    sorted_indices = np.argsort(-combined_value, axis=1)
    sorted_values = np.take_along_axis(combined_value, sorted_indices, axis=1)
    
    # Sort the ads and organic values separately
    sorted_ads_indices = np.argsort(-value_ads_adjusted, axis=1)
    sorted_ads_values = np.take_along_axis(value_ads_adjusted, sorted_ads_indices, axis=1)
    sorted_org_indices = np.argsort(-value_org_adjusted, axis=1)
    sorted_org_values = np.take_along_axis(value_org_adjusted, sorted_org_indices, axis=1)
    
    # Initialize queues for ads and organic results
    ads_queue = np.zeros([value_ads.shape[0], 2])
    org_queue = np.zeros([value_ads.shape[0], 2])
    
    # Fill the queues based on sorted values
    for i in range(combined_value.shape[0]):
        ads_pos, org_pos = 0, 0
        for j in range(4):
            if sorted_values[i, j] in value_ads_adjusted[i]:
                if ads_queue[i, -1] == 0:
                    ads_queue[i, ads_pos] = j
                    ads_pos += 1
                else:
                    org_queue[i, org_pos] = j
                    org_pos += 1
            else:
                if org_queue[i, -1] == 0:
                    org_queue[i, org_pos] = j
                    org_pos += 1
                else:
                    ads_queue[i, ads_pos] = j
                    ads_pos += 1
    
    # Sort CTR and CVR values based on sorted indices
    sorted_ctr_ads = np.take_along_axis(ctr_ads, sorted_ads_indices, axis=1)
    sorted_ctr_org = np.take_along_axis(ctr_org, sorted_org_indices, axis=1)
    sorted_cvr_ads = np.take_along_axis(cvr_ads, sorted_ads_indices, axis=1)
    sorted_cvr_org = np.take_along_axis(cvr_org, sorted_org_indices, axis=1)
    
    # Calculate clicks and CVR
    click = np.mean(np.sum(sorted_ctr_ads[:, :2] * get_position_ctr(ads_queue[:, :2], 2), axis=1) + 
                    np.sum(sorted_ctr_org[:, :2] * get_position_ctr(org_queue[:, :2], 2), axis=1))
    cvr = np.mean(np.mean(sorted_cvr_ads[:, :2] / 2 + sorted_cvr_org[:, :2] / 2))
    
    # Calculate payments
    payment = np.zeros([value_ads.shape[0], 2])
    payment[:, 0] = ((sorted_ads_values[:, 1] * get_position_ctr(ads_queue[:, 0], 1) + 
                      sorted_ads_values[:, 2] * get_position_ctr(ads_queue[:, 1], 1) - 
                      sorted_ads_values[:, 1] * get_position_ctr(ads_queue[:, 1], 1)) - 
                     beta * sorted_cvr_ads[:, 0] - 
                     alpha * sorted_ctr_ads[:, 0] * get_position_ctr(ads_queue[:, 0], 1)) / (sorted_ctr_ads[:, 0] * get_position_ctr(ads_queue[:, 0], 1))
    payment[:, 1] = ((sorted_ads_values[:, 2] * get_position_ctr(ads_queue[:, 1], 1)) - 
                     beta * sorted_cvr_ads[:, 1] - 
                     alpha * sorted_ctr_ads[:, 1] * get_position_ctr(ads_queue[:, 1], 1)) / (sorted_ctr_ads[:, 1] * get_position_ctr(ads_queue[:, 1], 1))
    
    # Calculate cost
    cost = np.mean(np.sum(np.maximum(payment, 0) * sorted_ctr_ads[:, :2] * get_position_ctr(ads_queue[:, :2], 2), axis=1))
    
    return click, cost, cvr

def alpha_GSP_Mechanism(value_ads, ctr_ads, ctr_org, cvr_ads, cvr_org, alpha, beta):
    """
    Implements the alpha-GSP mechanism for ad auctions.
    
    Parameters:
    - value_ads: The value of ads.
    - ctr_ads: Click-through rate of ads.
    - ctr_org: Click-through rate of organic results.
    - cvr_ads: Conversion rate of ads.
    - cvr_org: Conversion rate of organic results.
    - alpha: Weight for CTR.
    - beta: Weight for CVR.
    
    Returns:
    - click: Average number of clicks.
    - cost: Average cost.
    - cvr: Average conversion rate.
    """
    # Calculate the value for ads and organic results
    value_ads_adjusted = value_ads * ctr_ads + alpha * ctr_ads + beta * cvr_ads
    value_org_adjusted = alpha * ctr_org + beta * cvr_org
    combined_value = np.hstack((value_ads_adjusted, value_org_adjusted))
    
    # Sort the combined values
    sorted_indices = np.argsort(-combined_value, axis=1)
    sorted_values = np.take_along_axis(combined_value, sorted_indices, axis=1)
    
    # Sort the ads and organic values separately
    sorted_ads_indices = np.argsort(-value_ads_adjusted, axis=1)
    sorted_ads_values = np.take_along_axis(value_ads_adjusted, sorted_ads_indices, axis=1)
    sorted_org_indices = np.argsort(-value_org_adjusted, axis=1)
    sorted_org_values = np.take_along_axis(value_org_adjusted, sorted_org_indices, axis=1)
    
    # Initialize queues for ads and organic results
    ads_queue = np.zeros([value_ads.shape[0], 2])
    org_queue = np.zeros([value_ads.shape[0], 2])
    
    # Fill the queues based on sorted values
    for i in range(combined_value.shape[0]):
        ads_pos, org_pos = 0, 0
        for j in range(4):
            if sorted_values[i, j] in value_ads_adjusted[i]:
                if ads_queue[i, -1] == 0:
                    ads_queue[i, ads_pos] = j
                    ads_pos += 1
                else:
                    org_queue[i, org_pos] = j
                    org_pos += 1
            else:
                if org_queue[i, -1] == 0:
                    org_queue[i, org_pos] = j
                    org_pos += 1
                else:
                    ads_queue[i, ads_pos] = j
                    ads_pos += 1
    
    # Sort CTR and CVR values based on sorted indices
    sorted_ctr_ads = np.take_along_axis(ctr_ads, sorted_ads_indices, axis=1)
    sorted_ctr_org = np.take_along_axis(ctr_org, sorted_org_indices, axis=1)
    sorted_cvr_ads = np.take_along_axis(cvr_ads, sorted_ads_indices, axis=1)
    sorted_cvr_org = np.take_along_axis(cvr_org, sorted_org_indices, axis=1)
    
    # Calculate clicks and CVR
    click = np.mean(np.sum(sorted_ctr_ads[:, :2] * get_position_ctr(ads_queue[:, :2], 2), axis=1) + 
                    np.sum(sorted_ctr_org[:, :2] * get_position_ctr(org_queue[:, :2], 2), axis=1))
    cvr = np.mean(np.mean(sorted_cvr_ads[:, :2] / 2 + sorted_cvr_org[:, :2] / 2))
    
    # Calculate payments
    payment = np.zeros([value_ads.shape[0], 2])
    payment[:, 0] = ((sorted_ads_values[:, 1] - 
                     alpha * sorted_ctr_ads[:, 0] - 
                     beta * sorted_cvr_ads[:, 0]) / 
                     sorted_ctr_ads[:, 0])
    payment[:, 1] = ((sorted_ads_values[:, 2] - 
                     alpha * sorted_ctr_ads[:, 1] - 
                     beta * sorted_cvr_ads[:, 1]) / 
                     sorted_ctr_ads[:, 1])
    
    # Calculate cost
    cost = np.mean(np.sum(np.maximum(payment, 0) * sorted_ctr_ads[:, :2] * get_position_ctr(ads_queue[:, :2], 2), axis=1))
    
    return click, cost, cvr

